--[[
Copyright (C) GtX (Andy), 2025

Author: GtX | Andy
Date: 27.04.2025
Revision: FS25-01

Contact:
https://forum.giants-software.com
https://github.com/GtX-Andy

Important:
Not to be added to any mods / maps or modified from its current release form.
No modifications may be made to this script, including conversion to other game versions without written permission from GtX | Andy
Copying or removing any part of this code for external use without written permission from GtX | Andy is prohibited.

Darf nicht zu Mods / Maps hinzugefügt oder von der aktuellen Release-Form geändert werden.
Ohne schriftliche Genehmigung von GtX | Andy dürfen keine Änderungen an diesem Skript vorgenommen werden, einschließlich der Konvertierung in andere Spielversionen
Das Kopieren oder Entfernen irgendeines Teils dieses Codes zur externen Verwendung ohne schriftliche Genehmigung von GtX | Andy ist verboten.
]]

PlaceableCustomFieldEvent = {}

local PlaceableCustomFieldEvent_mt = Class(PlaceableCustomFieldEvent, Event)
InitEventClass(PlaceableCustomFieldEvent, "PlaceableCustomFieldEvent")

function PlaceableCustomFieldEvent.emptyNew()
    return Event.new(PlaceableCustomFieldEvent_mt, NetworkNode.CHANNEL_MAIN)
end

function PlaceableCustomFieldEvent.new(placeable, fieldData, releasePlaceable)
    local self = PlaceableCustomFieldEvent.emptyNew()

    self.placeable = placeable
    self.fieldData = fieldData
    self.releasePlaceable = Utils.getNoNil(releasePlaceable, fieldData == nil)

    return self
end

function PlaceableCustomFieldEvent:readStream(streamId, connection)
    self.placeable = NetworkUtil.readNodeObject(streamId)
    self.releasePlaceable = streamReadBool(streamId)

    if not self.releasePlaceable then
        local numVertices = streamReadUInt16(streamId)

        local fieldData = {
            polygon = Polygon2D.new(numVertices),
            fruitTypeIndex = FruitType.UNKNOWN,
            groundTypeIndex = FieldGroundType.PLOWED
        }

        for i = 1, numVertices do
            fieldData.polygon:addPos(streamReadFloat32(streamId), streamReadFloat32(streamId))
        end

        fieldData.fruitTypeIndex = streamReadUIntN(streamId, FruitTypeManager.SEND_NUM_BITS)
        fieldData.groundTypeIndex = FieldGroundType.readStream(streamId)

        fieldData.totalCost = streamReadInt32(streamId)

        self.fieldData = fieldData
    end

    self:run(connection)
end

function PlaceableCustomFieldEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.placeable)

    if not streamWriteBool(streamId, self.releasePlaceable) then
        local fieldData = self.fieldData

        streamWriteUInt16(streamId, fieldData.polygon:getNumVertices())

        local numVertices = fieldData.polygon:getVertices()

        for _, vertex in ipairs(numVertices) do
            streamWriteFloat32(streamId, vertex)
        end

        local fruitTypeIndex = fieldData.fruitTypeIndex or FruitType.UNKNOWN
        streamWriteUIntN(streamId, fruitTypeIndex, FruitTypeManager.SEND_NUM_BITS)

        local groundTypeIndex = fieldData.groundTypeIndex

        if groundTypeIndex == nil then
            groundTypeIndex = fruitTypeIndex ~= FruitType.UNKNOWN and FieldGroundType.CULTIVATED or FieldGroundType.PLOWED
        end

        FieldGroundType.writeStream(streamId, groundTypeIndex)

        streamWriteInt32(streamId, fieldData.totalCost or 0)
    end
end

function PlaceableCustomFieldEvent:run(connection)
    local placeable = self.placeable

    if self.releasePlaceable then
        if placeable ~= nil then
            placeable:onSell()
            placeable:delete()
        else
            Logging.info("Failed to release placeable because it does not exist, request has been ignored.")
        end

        return
    end

    local fieldData = self.fieldData

    if fieldData ~= nil and placeable ~= nil then
        local densityMapPolygon = DensityMapPolygon.new()
        densityMapPolygon:updateFromPolygon2D(fieldData.polygon)

        local fieldUpdateTask = FieldUpdateTask.new()
        local oldSetFinished = fieldUpdateTask.setFinished

        function fieldUpdateTask.setFinished(task, ...)
            oldSetFinished(task, ...)

            if placeable ~= nil and placeable:getIsSynchronized() then
                connection:sendEvent(PlaceableCustomFieldAnswerEvent.new(placeable, PlaceableRiceField.BUILD_STATUS.OK))
            end

            local totalCost = fieldData.totalCost or 0

            if totalCost > 0 then
                local farmId = g_currentMission:getFarmId(connection)

                if farmId ~= nil then
                    g_currentMission:addMoney(-totalCost, farmId, MoneyType.PROPERTY_MAINTENANCE, true, true)
                end
            end
        end

        fieldUpdateTask:setName("PlaceableCustomField - Create Field")
        fieldUpdateTask:setArea(densityMapPolygon)
        fieldUpdateTask:clearHeight()

        fieldUpdateTask:setWeedState(0)
        fieldUpdateTask:setStoneLevel(0)
        fieldUpdateTask:setLimeLevel(0)
        fieldUpdateTask:setPlowLevel(1)

        if fieldData.fruitTypeIndex ~= nil and fieldData.fruitTypeIndex ~= FruitType.UNKNOWN then
            fieldUpdateTask:setFruit(fieldData.fruitTypeIndex, 1)
        else
            fieldUpdateTask:setFruit(FruitType.UNKNOWN, 0)
        end

        fieldUpdateTask:setSprayType(FieldSprayType.NONE)
        fieldUpdateTask:setGroundType(fieldData.groundTypeIndex)

        fieldUpdateTask:resetDisplacement()
        fieldUpdateTask:clearTireTracks()

        fieldUpdateTask:enqueue()
    end
end
