-- Network event for upgrading a ProductionPoint

UpgradeFactoryEvent = {}
local UpgradeFactoryEvent_mt = Class(UpgradeFactoryEvent, Event)

InitEventClass(UpgradeFactoryEvent, "UpgradeFactoryEvent")

function UpgradeFactoryEvent.emptyNew()
    return Event.new(UpgradeFactoryEvent_mt)
end

function UpgradeFactoryEvent.new(pp, newLevel, cost, farmId)
    local self = UpgradeFactoryEvent.emptyNew()
    self.pp = pp
    self.newLevel = newLevel
    self.cost = cost
    self.farmId = farmId
    return self
end

function UpgradeFactoryEvent:writeStream(streamId, connection)
    NetworkUtil.writeNodeObject(streamId, self.pp)
    streamWriteUIntN(streamId, self.newLevel, 4)
    streamWriteInt32(streamId, self.cost)
    streamWriteUIntN(streamId, self.farmId or 0, FarmManager.FARM_ID_SEND_NUM_BITS)
end

function UpgradeFactoryEvent:readStream(streamId, connection)
    self.pp = NetworkUtil.readNodeObject(streamId)
    self.newLevel = streamReadUIntN(streamId, 4)
    self.cost = streamReadInt32(streamId)
    self.farmId = streamReadUIntN(streamId, FarmManager.FARM_ID_SEND_NUM_BITS)
    self:run(connection)
end

function UpgradeFactoryEvent:run(connection)
    -- Server authoritative
    if not connection:getIsServer() then
        -- From client to server
        self:_applyAndBroadcast(connection)
    else
        -- From server to clients
        UpgradableFactories.setLevel(self.pp, self.newLevel)
    end
end

function UpgradeFactoryEvent:_applyAndBroadcast(connection)
    if self.pp == nil then return end
    local mission = g_currentMission
    local farmId = self.farmId or mission:getFarmId()

    -- Validate ownership
    if self.pp.getOwnerFarmId ~= nil and self.pp:getOwnerFarmId() ~= farmId then
        return
    end

    -- Validate funds
    if mission:getMoney(farmId) < self.cost then
        g_currentMission:showBlinkingWarning(g_i18n:getText("ui_upgradeFactoryNoMoney"), 3000)
        return
    end

    -- Deduct and apply
    mission:addMoney(-self.cost, farmId, MoneyType.OTHER, true)
    UpgradableFactories.setLevel(self.pp, self.newLevel)

    if g_server ~= nil then
        g_server:broadcastEvent(UpgradeFactoryEvent.new(self.pp, self.newLevel, self.cost, farmId), nil, connection, self.pp)
    end
end

function UpgradeFactoryEvent.sendEvent(pp, newLevel, cost, farmId)
    if g_server ~= nil then
        -- Singleplayer or server-side
        local event = UpgradeFactoryEvent.new(pp, newLevel, cost, farmId)
        event:_applyAndBroadcast(nil)
    else
        -- Client sends to server
        g_client:getServerConnection():sendEvent(UpgradeFactoryEvent.new(pp, newLevel, cost, farmId))
    end
end
